import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Briefcase, User, Star, TrendingUp, Award, Target, BarChart3, CheckCircle2, ArrowRight, Calendar, TrendingDown } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MySessions.css'

const MySessions = () => {
  const [sessions, setSessions] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('careerSessions')
    if (saved) {
      try {
        setSessions(JSON.parse(saved))
      } catch (e) {
        setSessions([])
      }
    } else {
      const defaultSessions = [
        {
          id: 1,
          consultant: 'Елена Смирнова',
          topic: 'Подготовка резюме',
          result: 'Резюме обновлено, получено 3 приглашения на собеседование',
          review: 'Отличная консультация, помогла структурировать опыт',
          rating: 5,
          date: '2025-03-25'
        },
        {
          id: 2,
          consultant: 'Дмитрий Петров',
          topic: 'Собеседование',
          result: 'Успешно прошел собеседование, получил оффер',
          review: 'Профессиональная подготовка, все советы пригодились',
          rating: 5,
          date: '2025-03-20'
        },
        {
          id: 3,
          consultant: 'Анна Козлова',
          topic: 'Смена профессии',
          result: 'Определен план перехода в новую сферу',
          review: 'Помогла понять, куда двигаться дальше',
          rating: 4,
          date: '2025-03-15'
        }
      ]
      setSessions(defaultSessions)
      localStorage.setItem('careerSessions', JSON.stringify(defaultSessions))
    }
  }, [])

  const totalSessions = sessions.length
  const averageRating = sessions.length > 0
    ? (sessions.reduce((sum, s) => sum + s.rating, 0) / sessions.length).toFixed(1)
    : 0
  const uniqueConsultants = new Set(sessions.map(s => s.consultant)).size
  const successfulResults = sessions.filter(s => s.result.toLowerCase().includes('успешно') || s.result.toLowerCase().includes('получен')).length

  const progressPercentage = totalSessions > 0 ? Math.min((successfulResults / totalSessions) * 100, 100) : 0

  return (
    <div className="my-sessions">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7 }}
        >
          <div className="hero-content">
            <motion.div 
              className="hero-icon-wrapper"
              initial={{ scale: 0, rotate: -180 }}
              animate={{ scale: 1, rotate: 0 }}
              transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
            >
              <Briefcase size={100} strokeWidth={2.5} className="hero-icon" />
              <div className="hero-icon-glow"></div>
            </motion.div>
            <div className="hero-text">
              <h1 className="hero-title">Мои сессии</h1>
              <p className="hero-description">
                Визуализируйте свой карьерный путь и отслеживайте прогресс
              </p>
            </div>
          </div>
        </motion.section>

        <motion.section 
          className="stats-progress-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <div className="progress-container">
            <div className="progress-header">
              <h3 className="progress-title">Прогресс карьерного роста</h3>
              <span className="progress-percentage">{Math.round(progressPercentage)}%</span>
            </div>
            <div className="progress-bar-wrapper">
              <motion.div 
                className="progress-bar"
                initial={{ width: 0 }}
                animate={{ width: `${progressPercentage}%` }}
                transition={{ duration: 1, delay: 0.6 }}
              >
                <div className="progress-bar-glow"></div>
              </motion.div>
            </div>
            <div className="progress-stats">
              <div className="progress-stat-item">
                <div className="progress-stat-icon" style={{ background: 'rgba(37, 99, 235, 0.15)' }}>
                  <Briefcase size={32} strokeWidth={2.5} color="#2563EB" />
                </div>
                <div className="progress-stat-content">
                  <div className="progress-stat-value">{totalSessions}</div>
                  <div className="progress-stat-label">Всего сессий</div>
                </div>
              </div>
              <div className="progress-stat-item">
                <div className="progress-stat-icon" style={{ background: 'rgba(16, 185, 129, 0.15)' }}>
                  <TrendingUp size={32} strokeWidth={2.5} color="#10B981" />
                </div>
                <div className="progress-stat-content">
                  <div className="progress-stat-value">{successfulResults}</div>
                  <div className="progress-stat-label">Успешных результатов</div>
                </div>
              </div>
              <div className="progress-stat-item">
                <div className="progress-stat-icon" style={{ background: 'rgba(6, 182, 212, 0.15)' }}>
                  <Star size={32} strokeWidth={2.5} color="#06B6D4" />
                </div>
                <div className="progress-stat-content">
                  <div className="progress-stat-value">{averageRating}</div>
                  <div className="progress-stat-label">Средняя оценка</div>
                </div>
              </div>
              <div className="progress-stat-item">
                <div className="progress-stat-icon" style={{ background: 'rgba(139, 92, 246, 0.15)' }}>
                  <User size={32} strokeWidth={2.5} color="#8B5CF6" />
                </div>
                <div className="progress-stat-content">
                  <div className="progress-stat-value">{uniqueConsultants}</div>
                  <div className="progress-stat-label">Консультантов</div>
                </div>
              </div>
            </div>
          </div>
        </motion.section>

        <ImageSlider images={[
          '/images/career-1.jpg',
          '/images/career-2.jpg',
          '/images/career-3.jpg',
          '/images/career-4.jpg'
        ]} />

        <motion.section 
          className="sessions-timeline-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <div className="section-header-split">
            <h2 className="section-title">История консультаций</h2>
            <div className="section-subtitle">Ваш карьерный путь</div>
          </div>
          {sessions.length > 0 ? (
            <div className="timeline-container">
              <div className="timeline-line"></div>
              {sessions.map((session, index) => (
                <motion.div
                  key={session.id}
                  className="timeline-item"
                  initial={{ opacity: 0, x: index % 2 === 0 ? -50 : 50 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.15, duration: 0.6 }}
                  whileHover={{ scale: 1.03, transition: { duration: 0.3 } }}
                >
                  <div className="timeline-marker">
                    <div className="timeline-dot"></div>
                    <div className="timeline-pulse"></div>
                  </div>
                  <div className={`timeline-content ${index % 2 === 0 ? 'left' : 'right'}`}>
                    <div className="timeline-card">
                      <div className="timeline-card-header">
                        <div className="timeline-topic-badge" style={{ background: `linear-gradient(135deg, #2563EB 0%, #3B82F6 100%)` }}>
                          {session.topic}
                        </div>
                        <div className="timeline-rating">
                          {Array.from({ length: 5 }, (_, i) => (
                            <Star
                              key={i}
                              size={16}
                              fill={i < session.rating ? '#2563EB' : 'none'}
                              stroke={i < session.rating ? '#2563EB' : '#CBD5E1'}
                              strokeWidth={2}
                            />
                          ))}
                        </div>
                      </div>
                      <div className="timeline-card-body">
                        <div className="timeline-consultant">
                          <User size={20} strokeWidth={2} color="#10B981" />
                          <span>{session.consultant}</span>
                        </div>
                        <div className="timeline-result-box">
                          <CheckCircle2 size={20} strokeWidth={2} color="#10B981" />
                          <div className="result-text-wrapper">
                            <span className="result-label">Достижение:</span>
                            <span className="result-text">{session.result}</span>
                          </div>
                        </div>
                        {session.review && (
                          <div className="timeline-review-box">
                            <Star size={18} strokeWidth={2} color="#F59E0B" fill="#F59E0B" />
                            <span className="review-text">{session.review}</span>
                          </div>
                        )}
                      </div>
                      <div className="timeline-date">
                        <Calendar size={16} strokeWidth={2} />
                        {new Date(session.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' })}
                      </div>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state-modern">
              <div className="empty-icon-modern">
                <Briefcase size={80} strokeWidth={1.5} />
                <div className="empty-icon-ring"></div>
              </div>
              <h3 className="empty-title">Начните свой карьерный путь</h3>
              <p className="empty-text">Запишитесь на первую консультацию и увидите её здесь</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="achievements-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.6 }}
        >
          <div className="achievements-header">
            <h2 className="section-title">Ключевые преимущества</h2>
            <p className="section-description">Что делает нас лучшими в карьерном консультировании</p>
          </div>
          <div className="achievements-grid">
            {[
              { icon: Target, title: 'Стратегическое планирование', text: 'Разрабатываем индивидуальные планы карьерного роста', color: '#2563EB', gradient: 'linear-gradient(135deg, #2563EB 0%, #3B82F6 100%)' },
              { icon: BarChart3, title: 'Измеримый прогресс', text: 'Отслеживаем каждый шаг вашего профессионального развития', color: '#10B981', gradient: 'linear-gradient(135deg, #10B981 0%, #34D399 100%)' },
              { icon: Award, title: 'Экспертный подход', text: 'Работаем только с сертифицированными карьерными консультантами', color: '#06B6D4', gradient: 'linear-gradient(135deg, #06B6D4 0%, #22D3EE 100%)' },
              { icon: TrendingUp, title: 'Долгосрочный результат', text: 'Формируем навыки для самостоятельного карьерного роста', color: '#8B5CF6', gradient: 'linear-gradient(135deg, #8B5CF6 0%, #A78BFA 100%)' }
            ].map((achievement, index) => {
              const IconComponent = achievement.icon
              return (
                <motion.div
                  key={achievement.title}
                  className="achievement-block"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ delay: 0.7 + index * 0.1, duration: 0.5 }}
                  whileHover={{ y: -8, scale: 1.02, transition: { duration: 0.3 } }}
                >
                  <div className="achievement-icon-wrapper" style={{ background: achievement.gradient }}>
                    <IconComponent size={36} strokeWidth={2.5} color="white" />
                    <div className="achievement-icon-glow"></div>
                  </div>
                  <div className="achievement-content">
                    <h3 className="achievement-title">{achievement.title}</h3>
                    <p className="achievement-text">{achievement.text}</p>
                  </div>
                  <div className="achievement-arrow">
                    <ArrowRight size={24} strokeWidth={2.5} color={achievement.color} />
                  </div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MySessions
